/*global define */
/*jslint white: true */

/*
	This class provides the 3x3 matrix implementation.

	It assumes that matrix M transforms point p as if it were a column vector: p' = M p
	It represents matrix as a column-major array: first entries correspond to entries in the first column.
*/

define(["lodash", "src/utils", "src/math/Vec2"],
function (lodash, utils, vec2) {
	"use strict";

	var kFloatEps = Math.pow(2, -23), // default to float eps -- 24 bits of precision: 2 ^ -23
	// mat3 objects will inherit array methods using prototype chain injection.
	// see rationale: http://perfectionkills.com/how-ecmascript-5-still-does-not-allow-to-subclass-an-array/to type injection works well.
		mat3_prototype = [],
		proto = "__proto__";

	function mat3 (object) {
		object = object || [1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0];
		object[proto] = mat3_prototype;
		return object;
	}

	/*
	 * The following matrix methods will be available as free functions (static/class functions) of mat3 type.
	 * We define these first and then use them to implement prototype methods for mat3 objects (instances of mat3 type).
	 * Use them as convenient and when efficiency really matters.
	 */

	// initializations
	function initWithEntries (m00, m10, m20, m01, m11, m21, m02, m12, m22, result) {
		result = result || mat3();
		result[0] = m00;
		result[1] = m10;
		result[2] = m20;
		result[3] = m01;
		result[4] = m11;
		result[5] = m21;
		result[6] = m02;
		result[7] = m12;
		result[8] = m22;
		return result;
	}
	mat3.initWithEntries = initWithEntries;

	function initWithArray (arr, result) {
		result = result || mat3();
		result[0] = arr[0];
		result[1] = arr[1];
		result[2] = arr[2];
		result[3] = arr[3];
		result[4] = arr[4];
		result[5] = arr[5];
		result[6] = arr[6];
		result[7] = arr[7];
		result[8] = arr[8];
		return result;
	}
	mat3.initWithArray = initWithArray;

	// TODO: remove after we retire Matrix2d and switch serialization format.
	// note: it transposes the matrix when applied to a row-major AofA (i.e. Matrix2d.js)
	// this is what we need when we switch back and forth between Ae and Eigen conventions. 
	function initWithArrayOfArrays (data, result) {
		result = result || mat3();
		result[0] = data[0][0];
		result[1] = data[0][1];
		result[2] = data[0][2];
		result[3] = data[1][0];
		result[4] = data[1][1];
		result[5] = data[1][2];
		result[6] = data[2][0];
		result[7] = data[2][1];
		result[8] = data[2][2];
		return result;
	}
	mat3.initWithArrayOfArrays = initWithArrayOfArrays;

	function from (arr) {
		return initWithArray(arr, mat3());
	}
	mat3.from = from;

	function clone (mat) {
		return mat.slice(0, 9);
	}
	mat3.clone = clone;

	function multiply (matA, matB, result) {
		var a00 = matA[0], a10 = matA[1], a20 = matA[2],
			a01 = matA[3], a11 = matA[4], a21 = matA[5],
			a02 = matA[6], a12 = matA[7], a22 = matA[8],

			b00 = matB[0], b10 = matB[1], b20 = matB[2],
			b01 = matB[3], b11 = matB[4], b21 = matB[5],
			b02 = matB[6], b12 = matB[7], b22 = matB[8];

		result = result || mat3();
		result[0] = a00 * b00 + a01 * b10 + a02 * b20;
		result[1] = a10 * b00 + a11 * b10 + a12 * b20;
		result[2] = a20 * b00 + a21 * b10 + a22 * b20;
		result[3] = a00 * b01 + a01 * b11 + a02 * b21;
		result[4] = a10 * b01 + a11 * b11 + a12 * b21;
		result[5] = a20 * b01 + a21 * b11 + a22 * b21;
		result[6] = a00 * b02 + a01 * b12 + a02 * b22;
		result[7] = a10 * b02 + a11 * b12 + a12 * b22;
		result[8] = a20 * b02 + a21 * b12 + a22 * b22;

		return result;
	}
	mat3.multiply = multiply;

	function identity (result) {
		result = mat3();
		return result;
	}
	mat3.identity = identity;

	function zero (result) {
		if (result) {
			result[1] = 0.0;
			result[2] = 0.0;
			result[3] = 0.0;
			result[5] = 0.0;
			result[6] = 0.0;
			result[7] = 0.0;
		} else {
			result = mat3();
		}
		result[0] = 0.0;
		result[4] = 0.0;
		result[8] = 0.0;
		return result;
	}
	mat3.zero = zero;

	function scaleAdd (a, matA, b, matB, result) {
		result = result || mat3();
		result[0] = a * matA[0] + b * matB[0];
		result[1] = a * matA[1] + b * matB[1];
		result[2] = a * matA[2] + b * matB[2];
		result[3] = a * matA[3] + b * matB[3];
		result[4] = a * matA[4] + b * matB[4];
		result[5] = a * matA[5] + b * matB[5];
		result[6] = a * matA[6] + b * matB[6];
		result[7] = a * matA[7] + b * matB[7];
		result[8] = a * matA[8] + b * matB[8];
		return result;
	}
	mat3.scaleAdd = scaleAdd;

	/**
	 * Initialize translation matrix
	 * @param position - array of translation displacements
	 * @param result - translation matrix
	 */
	function translation (position, result) {
		if (result) {
			result[0] = 1.0;
			result[1] = 0.0;
			result[2] = 0.0;
			result[3] = 0.0;
			result[4] = 1.0;
			result[5] = 0.0;
			result[8] = 1.0;
		} else {
			result = mat3();
		}
		result[6] = position[0];
		result[7] = position[1];

		return result;
	}
	mat3.translation = translation;

	function transpose (mat, result) {
		result = result || mat3();
		if (result === mat) {
			var a10 = mat[1], a20 = mat[2], a21 = mat[5];
			result[1] = mat[3];
			result[2] = mat[6];
			result[3] = a10;
			result[5] = mat[7];
			result[6] = a20;
			result[7] = a21;
		} else {
			result[0] = mat[0];
			result[1] = mat[3];
			result[2] = mat[6];
			result[3] = mat[1];
			result[4] = mat[4];
			result[5] = mat[7];
			result[6] = mat[2];
			result[7] = mat[5];
			result[8] = mat[8];
		}
		return result;
	}
	mat3.transpose = transpose;

	/**
	 * Initialize rotation matrix.
	 * @param amount - angle of rotation in radians
	 * @param result - rotation matrix
	 */
	function rotation (angle, result) {
		var c = Math.cos(angle),
			s = Math.sin(angle);

		if (result) {
			result[2] = 0.0;
			result[5] = 0.0;
			result[6] = 0.0;
			result[7] = 0.0;
			result[8] = 1.0;
		} else {
			result = mat3();
		}
		result[0] = c;
		result[1] = s;
		result[3] = -s;
		result[4] = c;

		return result;
	}
	mat3.rotation = rotation;

	/**
	 * Initialize scaling matrix.
	 * @param amount - scaling factors array
	 * @param result - scaling matrix
	 */
	function scaling (scale, result) {
		if (result) {
			result[1] = 0.0;
			result[2] = 0.0;
			result[3] = 0.0;
			result[5] = 0.0;
			result[6] = 0.0;
			result[7] = 0.0;
			result[8] = 1.0;
		} else {
			result = mat3();
		}
		result[0] = scale[0];
		result[4] = scale[1];

		return result;
	}
	mat3.scaling = scaling;

	/**
	 * Initialize skewing matrix.
	 * @param amount - skew amount
	 * @param axis - skew angle in radians
	 * @param resutl - skewing matrix
	 */
	function skewing (amount, axis, result) {
		var skew_amt = Math.tan(amount);

		if (skew_amt === Infinity) {
			throw new Error("could not compute skewing matrix.");
		}

		if (result) {
			result[0] = 1.0;
			result[1] = 0.0;
			result[2] = 0.0;
			result[4] = 1.0;
			result[5] = 0.0;
			result[6] = 0.0;
			result[7] = 0.0;
			result[8] = 1.0;
		} else {
			result = mat3();
		}
		result[3] = -skew_amt;

		multiply(rotation(axis), result, result);
		multiply(result, rotation(-axis), result);
		return result;
	}
	mat3.skewing = skewing;

	/** Assembles affine matrix
	 *  No y-shear so that all transforms with positive scaling map back to same params.
	 *
	 */
	function affine (position, scale, shear, angle, result) {
		utils.assert(shear.length === 1, "affine expects only single x-shear component.  update call site.");

		var c = Math.cos(angle),
			s = Math.sin(angle),
			px = position[0], py = position[1],
			scaleX = scale[0], scaleY = scale[1],
			shearX = shear[0];

		if (result) {
			result[2] = 0.0;
			result[5] = 0.0;
			result[8] = 1.0;
		} else {
			result = mat3();
		}
		
		result[0] = c * scaleX;
		result[1] = s * scaleX;
		result[3] = (c * shearX - s) * scaleY;
		result[4] = (s * shearX + c) * scaleY;
		result[6] = px;
		result[7] = py;

		return result;
	}
	mat3.affine = affine;

	/**
	 * Set translation for an affine matrix.
	 * @param translation - array with new translation values
	 * @param result - location of affine matrix to modify
	 * @return affine matrix with the new translation.
	 */
	function setTranslation (translation, result) {
		result[6] = translation[0];
		result[7] = translation[1];
		return result;
	}
	mat3.setTranslation = setTranslation;

	/**
	 * Get translation from an affine matrix.
	 * @param mat - affine matrix to access
	 * @param result - location for the translation result
	 * @return array with the translation.
	 */
	function getTranslation (mat, result) {
		result = result || vec2();
		result[0] = mat[6];
		result[1] = mat[7];
		return result;
	}
	mat3.getTranslation = getTranslation;

	/**
	 * Get the scale squared from an affine matrix.
	 * @param mat - affine matrix to access
	 * @param result - location for the translation result
	 * @return array with the scale vector.
	 */
	function getScaleSquared (mat, result) {
		result = result || vec2();
		result[0] = mat[0]*mat[0] + mat[1]*mat[1] + mat[2]*mat[2];
		result[1] = mat[3]*mat[3] + mat[4]*mat[4] + mat[5]*mat[5];
		return result;
	}
	mat3.getScaleSquared = getScaleSquared;

	/**
	 * Get the scale from an affine matrix.
	 * @param mat - affine matrix to access
	 * @param result - location for the translation result
	 * @return array with the scale vector.
	 */
	function getScale (mat, result) {
		result = getScaleSquared(mat, result);
		
		result[0] = Math.sqrt(result[0]);
		result[1] = Math.sqrt(result[1]);
		return result;
	}
	mat3.getScale = getScale;

	function polarRotation (linear, result) {
		var t = linear[0] + linear[3],
			d = linear[2] - linear[1],
			w, c, s;

		result = result || [];

		if (Math.abs(d) <= kFloatEps * Math.abs(t)) {
			// already symmetric: result = identity
			result[0] = 1;	result[2] = 0;
			result[1] = 0;	result[3] = 1;
			return result;
		} 

		w = Math.sqrt(t*t + d*d);
		c = t / w;
		s = d / w;

		result[0] = c;	result[2] = s;
		result[1] = -s;	result[3] = c;
		return result;
	}

	/** Check linear transform for singularity
	 */
	function isNonSingular (mat) {
		// linear transform is top-left 2x2 block
		return (Math.abs(mat[0]*mat[4] - mat[1]*mat[3]) >= kFloatEps);
	}
	mat3.isNonSingular = isNonSingular;

	/** Decompose affine matrix according to W3C standard: http://www.w3.org/TR/css-transforms-1/
	 */
	function decomposeAffine (mat, position, scale, shear, m2Rotation) {
		var m00, m01, m10, m11, scaleX, scaleY, shearXscaleY, shearX;

		// extract position: last column
		position[0] = mat[6];
		position[1] = mat[7];

		// extract linear transform: top-left 2x2 block
		m00 = mat[0]; m01 = mat[3];
		m10 = mat[1]; m11 = mat[4];

		if(Math.abs(m00*m11 - m10*m01) < kFloatEps)
		{
			// TODO: use this assert to catch these cases and verify error handling below
			// utils.assert(Math.abs(m00*m11 - m10*m01) >= kFloatEps, "decomposeAffine: singular matrix.");

			// no rotation
			m2Rotation[0] = 1; m2Rotation[2] = 0;
			m2Rotation[1] = 0; m2Rotation[3] = 1;

			// no shear
			shear[0] = 0;

			// small scale 
			scale[0] = Math.sqrt(m00*m00 + m10*m10);
			scale[1] = Math.sqrt(m01*m01 + m11*m11);

			return;
		}


		scaleX = Math.sqrt(m00*m00 + m10*m10);
		m00 = m00 / scaleX;
		m10 = m10 / scaleX;

		shearXscaleY = m00*m01 + m10*m11; 

		m01 = m01 - m00 * shearXscaleY;
		m11 = m11 - m10 * shearXscaleY;

		scaleY = Math.sqrt(m01*m01 + m11*m11);
		m01 = m01 / scaleY;
		m11 = m11 / scaleY;
		shearX = shearXscaleY  / scaleY;

		if(m00*m11 - m10*m01 < 0)
		{ // reflection!
			m00 = -m00;
			m10 = -m10;
			shearX = -shearX;
			scaleX = -scaleX;
		}

		m2Rotation[0] = m00; m2Rotation[2] = m01;
		m2Rotation[1] = m10; m2Rotation[3] = m11;

		scale[0] = scaleX;
		scale[1] = scaleY;

		shear[0] = shearX;
	}
	mat3.decomposeAffine = decomposeAffine;

	function removeScale (mat, result0) {
		var result = result0 || mat3(),
			m2Linear = [], 
			m2Rotation = [];

		// extract linear matrix component: top-left 2x2 block
		m2Linear[0] = mat[0];		m2Linear[2] = mat[3];
		m2Linear[1] = mat[1];		m2Linear[3] = mat[4];

		// compute rotation
		polarRotation(m2Linear, m2Rotation);

		// reconstitute without scale
		result[0] = m2Rotation[0];		result[3] = m2Rotation[2];		result[6] = mat[6];
		result[1] = m2Rotation[1];		result[4] = m2Rotation[3];		result[7] = mat[7];
		result[2] = mat[2];				result[5] = mat[5];				result[8] = mat[8];

		return result;
	}
	mat3.removeScale = removeScale;

	function invert (mat, result) {
		var a00 = mat[0], a10 = mat[1], a20 = mat[2],
			a01 = mat[3], a11 = mat[4], a21 = mat[5],
			a02 = mat[6], a12 = mat[7], a22 = mat[8],
			t00 = a11 * a22 - a12 * a21,
			t10 = a12 * a20 - a10 * a22,
			t20 = a10 * a21 - a11 * a20,
			det = a00 * t00 + a01 * t10 + a02 * t20,
			inv;

		if (Math.abs(det) < kFloatEps) {
			throw new Error("could not invert matrix.");
		}

		inv = 1 / det;
		result = result || mat3();
		result[0] = t00 * inv;
		result[3] = (a02 * a21 - a01 * a22) * inv;
		result[6] = (a01 * a12 - a02 * a11) * inv;

		result[1] = t10 * inv;
		result[4] = (a00 * a22 - a02 * a20) * inv;
		result[7] = (a02 * a10 - a00 * a12) * inv;

		result[2] = t20 * inv;
		result[5] = (a01 * a20 - a00 * a21) * inv;
		result[8] = (a00 * a11 - a01 * a10) * inv;
		return result;
	}
	mat3.invert = invert;

	function equals (matA, matB) {
		return matA[0] === matB[0]
		&& matA[1] === matB[1]
		&& matA[2] === matB[2]
		&& matA[3] === matB[3]
		&& matA[4] === matB[4]
		&& matA[5] === matB[5]
		&& matA[6] === matB[6]
		&& matA[7] === matB[7]
		&& matA[8] === matB[8];
	}
	mat3.equals = equals;

	function equalsApproximately (matA, matB, eps0) {
		// default to float eps
		var eps = eps0 || kFloatEps;

		return Math.abs(matA[0] - matB[0]) < eps
		&& Math.abs(matA[1] - matB[1]) < eps
		&& Math.abs(matA[2] - matB[2]) < eps
		&& Math.abs(matA[3] - matB[3]) < eps
		&& Math.abs(matA[4] - matB[4]) < eps
		&& Math.abs(matA[5] - matB[5]) < eps
		&& Math.abs(matA[6] - matB[6]) < eps
		&& Math.abs(matA[7] - matB[7]) < eps
		&& Math.abs(matA[8] - matB[8]) < eps;
	}
	mat3.equalsApproximately = equalsApproximately;

	/*
	 * The following matrix methods will be available as methods on mat3 objects (instances of mat3 type).
	 * They are primarily syntactic sugar which delegates to the above free functions.
	 * Use them as convenient but when efficiency really matters use free functions instead.
	 */

	function delegateTo (invoke) {
		return function (name) {
			var method = mat3[name];
			if (! method) { throw new Error("mat3: method not found."); }
			if (mat3_prototype[name]) { throw new Error("mat3: method already exists."); }
			mat3_prototype[name] = invoke(method);
		};
	}

	mat3_prototype.init = function (m00, m10, m20, m01, m11, m21, m02, m12, m22) {
		return initWithEntries(m00, m10, m20, m01, m11, m21, m02, m12, m22, this);
	};

	mat3_prototype.setIdentity = function () {
		return identity(this);
	};

	mat3_prototype.setZero = function () {
		return zero(this);
	};

	mat3_prototype.setTranslation = function (arg) {
		return setTranslation(arg, this);
	};

	lodash.forEach([
		"transpose", "invert"
	], delegateTo(function (method) {
		return function () {
			return method(this, mat3());
		};
	}));

	/**
	 * Combine with translation in the local frame.
	 * @param {array} position - displacement expressed relative to the local frame defined by this object.
	 * @return combined matrix: this * translation
	 */
	mat3_prototype.translate = function (position) {
		utils.assert(arguments.length === 1);
		var m = translation(position);
		return multiply(this, m, mat3());
	};

	/**
	 * Combine with translation in the canonical frame.
	 * @param {array} position - displacement expressed relative to the absolute frame.
	 * @return combined matrix: translation * this
	 */
	mat3_prototype.pretranslate = function (position) {
		utils.assert(arguments.length === 1);
		var m = translation(position);
		return multiply(m, this, mat3());
	};

	/**
	 * Combine with rotation in the local frame.
	 * @param angle - counter-clockwise angle relative to the local frame defined by this object.
	 * @return combined matrix: this * rotation
	 */
	mat3_prototype.rotate = function (angle) {
		utils.assert(arguments.length === 1);
		var m = rotation(angle);
		return multiply(this, m, mat3());
	};

	/**
	 * Combine with rotation in the canonical frame.
	 * @param angle - counter-clockwise angle relative to the absolute frame.
	 * @return combined matrix: rotation * this
	 */
	mat3_prototype.prerotate = function (angle) {
		utils.assert(arguments.length === 1);
		var m = rotation(angle);
		return multiply(m, this, mat3());
	};

	/**
	 * Combine with scaling in the local frame.
	 * @param scale - scaling factors relative to the local frame defined by this object.
	 * @return combined matrix: this * scaling
	 */
	mat3_prototype.scale = function (scale) {
		utils.assert(arguments.length === 1);
		var m = scaling(scale);
		return multiply(this, m, mat3());
	};

	/**
	 * Combine with scaling in the canonical frame.
	 * @param {array} scale - scaling factors relative to the absolute frame.
	 * @return combined matrix: scaling * this
	 */
	mat3_prototype.prescale = function (scale) {
		utils.assert(arguments.length === 1);
		var m = scaling(scale);
		return multiply(m, this, mat3());
	};

	/**
	 * Combine with skewing in the local frame.
	 * @param amount - skew amount relative to the local frame defined by this object.
	 * @param axis - counter-clockwise angle relative to the local frame defined by this object.
	 * @return combined matrix: this * skewing
	 */
	mat3_prototype.skew = function (amount, axis) {
		utils.assert(arguments.length === 2);
		var m = skewing(amount, axis);
		return multiply(this, m, mat3());
	};

	/**
	 * Combine with skewing in the canonical frame.
	 * @param amount - skew amount relative to the canonical frame.
	 * @param axis - counter-clockwise angle relative to the canonical frame.
	 * @return combined matrix: skewing * this
	 */
	mat3_prototype.preskew = function (amount, axis) {
		utils.assert(arguments.length === 1);
		var m = skewing(amount, axis);
		return multiply(m, this, mat3());
	};

	mat3_prototype.equals = function (arg) {
		return equals(this, arg);
	};

	mat3_prototype.equalsApproximately = function (arg, eps) {
		return equalsApproximately(this, arg, eps);
	};

	mat3_prototype.clone = function () {
		return mat3(this.slice(0, 9));
	};

	mat3_prototype.getTranslation = function (result0) {
		return getTranslation(this, result0);
	};

	return mat3;
});
